from pprint import pformat


class LinearTable:
    """
    >>> table = LinearTable()
    >>> print("Sprite" in table)
    False
    >>> table["Sprite"] = 91
    >>> print("Sprite" in table)
    True
    >>> table["Axe"] = 99
    >>> print(table["Axe"])
    99
    >>> table["Axe"] = 100 # Correcting a score
    >>> print(table["Axe"])
    100
    >>> print(table)
    LinearTable([('Sprite', 91), ('Axe', 100)])
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """

    def __init__(self):
        self._container = []

    def __setitem__(self, key, value):
        """Maps to: table[key] = value"""
        for i, (k, v) in enumerate(self._container):
            if k == key:
                self._container[i] = (key, value)
                return
        self._container.append((key, value))

    def __getitem__(self, key):
        """Maps to: value = table[key]"""
        for k, v in self._container:
            if k == key: return v
        raise KeyError(key)

    def __contains__(self, key):
        """Maps to: 'key' in table"""
        try:
            self[key]
            return True
        except KeyError:
            return False

    def __repr__(self):
        return f"LinearTable({self._container!r})"


class BinarySearchTable:
    """
    >>> table = BinarySearchTable()
    >>> print("Sprite" in table)
    False
    >>> table["Sprite"] = 91
    >>> print("Sprite" in table)
    True
    >>> table["Axe"] = 99
    >>> print(table["Axe"])
    99
    >>> table["Axe"] = 100 # Correcting a score
    >>> print(table["Axe"])
    100
    >>> print(table)
    BinarySearchTable([('Axe', 100), ('Sprite', 91)])
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """
    def __init__(self):
        self._container = []

    def _find_index(self, key):
        low = 0
        high = len(self._container) - 1
        while low <= high:
            mid = (low + high) // 2
            mid_key = self._container[mid][0]
            if mid_key == key: return mid, True
            elif mid_key < key: low = mid + 1
            else: high = mid - 1
        return low, False

    def __setitem__(self, key, value):
        idx, found = self._find_index(key)
        if found: self._container[idx] = (key, value)
        else: self._container.insert(idx, (key, value))

    def __getitem__(self, key):
        idx, found = self._find_index(key)
        if found: return self._container[idx][1]
        raise KeyError(key)

    def __contains__(self, key):
        try:
            self[key]
            return True
        except KeyError:
            return False

    def __repr__(self):
        return f"BinarySearchTable({self._container!r})"


class HashTable:
    """
    >>> table = HashTable(8)
    >>> print("Sprite" in table)
    False
    >>> table["Sprite"] = 91
    >>> print("Sprite" in table)
    True
    >>> table["Axe"] = 99
    >>> print(table["Axe"])
    99
    >>> table["Axe"] = 100 # Correcting a score
    >>> print(table["Axe"])
    100
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """
    def __init__(self, capacity=8):
        self._capacity = capacity
        self._buckets = [[] for _ in range(self._capacity)]
        self._size = 0

    def _get_index(self, key):
        return hash(key) % self._capacity

    def __setitem__(self, key, value):
        idx = self._get_index(key)
        bucket = self._buckets[idx]
        for i, (k, v) in enumerate(bucket):
            if k == key:
                bucket[i] = (key, value)
                return
        bucket.append((key, value))
        self._size += 1

    def __getitem__(self, key):
        idx = self._get_index(key)
        bucket = self._buckets[idx]
        for k, v in bucket:
            if k == key: return v
        raise KeyError(key)

    def __contains__(self, key):
        try:
            self[key]
            return True
        except KeyError:
            return False

    def __repr__(self):
        return (f"======== HashTable ========\n"
                f"Capacity: {self._capacity}, Size: {self._size}\n"
                f"Buckets:\n"
                f"{pformat(self._buckets, width=32)}\n"
                f"===========================")


class RehashableHashTable(HashTable):
    """
    >>> table = RehashableHashTable(4)
    >>> for index, letter in enumerate(["Axe", "Salt", "Sprite"]): table[letter] = index
    >>> table._capacity
    4
    >>> table["777"] = 3
    >>> table._capacity
    8
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """
    def __init__(self, capacity=8):
        self._load_factor_threshold = 0.75
        super().__init__(capacity)

    def __setitem__(self, key, value):
        if self._size / self._capacity >= self._load_factor_threshold:
            self._resize()
        h_code = hash(key)
        idx = h_code % self._capacity
        bucket = self._buckets[idx]
        for i, (h, k, v) in enumerate(bucket):
            if h == h_code and k == key:
                bucket[i] = (h, k, value)
                return
        bucket.append((h_code, key, value))
        self._size += 1

    def __getitem__(self, key):
        h_code = hash(key)
        idx = h_code % self._capacity
        bucket = self._buckets[idx]
        for h, k, v in bucket:
            if h == h_code and k == key:
                return v
        raise KeyError(key)

    def _resize(self):
        old_buckets = self._buckets
        self._capacity *= 2
        self._buckets = [[] for _ in range(self._capacity)]
        self._size = 0
        for bucket in old_buckets:
            for h, k, v in bucket:
                new_idx = h % self._capacity
                self._buckets[new_idx].append((h, k, v))
                self._size += 1


class OpenAddressingHashTable(RehashableHashTable):
    """
    >>> table = OpenAddressingHashTable(4)
    >>> for index, letter in enumerate(["Axe", "Salt", "Sprite"]): table[letter] = index
    >>> table._capacity
    4
    >>> table["777"] = 3
    >>> table._capacity
    8
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """
    def __init__(self, capacity=8):
        self._capacity = capacity
        self._buckets = [None] * self._capacity
        self._size = 0
        self._load_factor_threshold = 0.66

    def _get_slots(self, key):
        h_code = hash(key)
        capacity = self._capacity
        idx = h_code % capacity
        perturb = abs(h_code)
        while True:
            yield idx, h_code
            perturb >>= 5
            idx = (5 * idx + 1 + perturb) % capacity

    def __setitem__(self, key, value):
        if self._size / self._capacity >= self._load_factor_threshold:
            self._resize()
        for idx, h_code in self._get_slots(key):
            slot = self._buckets[idx]
            if slot is None:
                self._buckets[idx] = (h_code, key, value)
                self._size += 1
                return
            h, k, v = slot
            if h == h_code and k == key:
                self._buckets[idx] = (h, key, value)
                return

    def __getitem__(self, key):
        for idx, h_code in self._get_slots(key):
            slot = self._buckets[idx]
            if slot is None: raise KeyError(key)
            h, k, v = slot
            if h == h_code and k == key: return v

    def _resize(self):
        old_buckets = self._buckets
        self._capacity *= 2
        self._buckets = [None] * self._capacity
        self._size = 0
        for item in old_buckets:
            if item:
                h, k, v = item
                self.__setitem__(k, v)


class CompactHashTable(OpenAddressingHashTable):
    """
    >>> table = OpenAddressingHashTable(4)
    >>> for index, letter in enumerate(["Axe", "Salt", "Sprite"]): table[letter] = index
    >>> table._capacity
    4
    >>> table["777"] = 3
    >>> table._capacity
    8
    >>> del table["Sprite"]
    >>> print("Sprite" in table)
    False
    """
    def __init__(self, capacity=8):
        self._capacity = capacity
        self._indices = [-1] * self._capacity
        self._entries = []
        self._size = 0
        self._load_factor_threshold = 0.66

    def __setitem__(self, key, value):
        if self._size / self._capacity >= self._load_factor_threshold:
            self._resize()
        for idx, h_code in self._get_slots(key):
            entry_idx = self._indices[idx]
            if entry_idx == -1:
                self._indices[idx] = len(self._entries)
                self._entries.append([h_code, key, value])
                self._size += 1
                return
            h, k, v = self._entries[entry_idx]
            if h == h_code and k == key:
                self._entries[entry_idx][2] = value
                return

    def __getitem__(self, key):
        for idx, h_code in self._get_slots(key):
            entry_idx = self._indices[idx]
            if entry_idx == -1: raise KeyError(key)
            h, k, v = self._entries[entry_idx]
            if h == h_code and k == key: return v

    def _resize(self):
        old_entries = self._entries
        self._capacity *= 2
        self._indices = [-1] * self._capacity
        self._entries = []
        self._size = 0
        for h, k, v in old_entries:
            self.__setitem__(k, v)

    def __repr__(self):
        return (f"======== HashTable ========\n"
                f"Capacity: {self._capacity}, Size: {self._size}\n"
                f"Indices:\n"
                f"{pformat(self._indices, width=32)}\n"
                f"Entries:\n"
                f"{pformat(self._entries, width=32)}\n"
                f"===========================")
